from pathlib import Path

import pytest
import requests_mock

import tableauserverclient as TSC
from tableauserverclient import UserItem, GroupItem, PermissionsRule
from tableauserverclient.datetime_helpers import format_datetime, parse_datetime
from tableauserverclient.server.endpoint.exceptions import UnsupportedAttributeError

TEST_ASSET_DIR = Path(__file__).parent / "assets"

ADD_TAGS_XML = TEST_ASSET_DIR / "view_add_tags.xml"
GET_XML = TEST_ASSET_DIR / "view_get.xml"
GET_XML_ALL_FIELDS = TEST_ASSET_DIR / "view_get_all_fields.xml"
GET_XML_ID = TEST_ASSET_DIR / "view_get_id.xml"
GET_XML_USAGE = TEST_ASSET_DIR / "view_get_usage.xml"
GET_XML_ID_USAGE = TEST_ASSET_DIR / "view_get_id_usage.xml"
POPULATE_PREVIEW_IMAGE = TEST_ASSET_DIR / "Sample View Image.png"
POPULATE_PDF = TEST_ASSET_DIR / "populate_pdf.pdf"
POPULATE_CSV = TEST_ASSET_DIR / "populate_csv.csv"
POPULATE_EXCEL = TEST_ASSET_DIR / "populate_excel.xlsx"
POPULATE_PERMISSIONS_XML = TEST_ASSET_DIR / "view_populate_permissions.xml"
UPDATE_PERMISSIONS = TEST_ASSET_DIR / "view_update_permissions.xml"
UPDATE_XML = TEST_ASSET_DIR / "workbook_update.xml"


@pytest.fixture(scope="function")
def server():
    """Fixture to create a TSC.Server instance for testing."""
    server = TSC.Server("http://test", False)

    # Fake signin
    server._site_id = "dad65087-b08b-4603-af4e-2887b8aafc67"
    server._auth_token = "j80k54ll2lfMZ0tv97mlPvvSCRyD0DOM"
    server.version = "3.2"

    return server


def test_get(server: TSC.Server) -> None:
    response_xml = GET_XML.read_text()
    with requests_mock.mock() as m:
        m.get(server.views.baseurl, text=response_xml)
        all_views, pagination_item = server.views.get()

    assert 2 == pagination_item.total_available
    assert "d79634e1-6063-4ec9-95ff-50acbf609ff5" == all_views[0].id
    assert "ENDANGERED SAFARI" == all_views[0].name
    assert "SafariSample/sheets/ENDANGEREDSAFARI" == all_views[0].content_url
    assert "3cc6cd06-89ce-4fdc-b935-5294135d6d42" == all_views[0].workbook_id
    assert "5de011f8-5aa9-4d5b-b991-f462c8dd6bb7" == all_views[0].owner_id
    assert "5241e88d-d384-4fd7-9c2f-648b5247efc5" == all_views[0].project_id
    assert {"tag1", "tag2"} == all_views[0].tags
    assert all_views[0].created_at is None
    assert all_views[0].updated_at is None
    assert all_views[0].sheet_type is None

    assert "fd252f73-593c-4c4e-8584-c032b8022adc" == all_views[1].id
    assert "Overview" == all_views[1].name
    assert "Superstore/sheets/Overview" == all_views[1].content_url
    assert "6d13b0ca-043d-4d42-8c9d-3f3313ea3a00" == all_views[1].workbook_id
    assert "5de011f8-5aa9-4d5b-b991-f462c8dd6bb7" == all_views[1].owner_id
    assert "5b534f74-3226-11e8-b47a-cb2e00f738a3" == all_views[1].project_id
    assert "2002-05-30T09:00:00Z" == format_datetime(all_views[1].created_at)
    assert "2002-06-05T08:00:59Z" == format_datetime(all_views[1].updated_at)
    assert "story" == all_views[1].sheet_type


def test_get_by_id(server: TSC.Server) -> None:
    response_xml = GET_XML_ID.read_text()
    with requests_mock.mock() as m:
        m.get(server.views.baseurl + "/d79634e1-6063-4ec9-95ff-50acbf609ff5", text=response_xml)
        view = server.views.get_by_id("d79634e1-6063-4ec9-95ff-50acbf609ff5")

    assert "d79634e1-6063-4ec9-95ff-50acbf609ff5" == view.id
    assert "ENDANGERED SAFARI" == view.name
    assert "SafariSample/sheets/ENDANGEREDSAFARI" == view.content_url
    assert "3cc6cd06-89ce-4fdc-b935-5294135d6d42" == view.workbook_id
    assert "5de011f8-5aa9-4d5b-b991-f462c8dd6bb7" == view.owner_id
    assert "5241e88d-d384-4fd7-9c2f-648b5247efc5" == view.project_id
    assert {"tag1", "tag2"} == view.tags
    assert "2002-05-30T09:00:00Z" == format_datetime(view.created_at)
    assert "2002-06-05T08:00:59Z" == format_datetime(view.updated_at)
    assert "story" == view.sheet_type


def test_get_by_id_usage(server: TSC.Server) -> None:
    response_xml = GET_XML_ID_USAGE.read_text()
    with requests_mock.mock() as m:
        m.get(
            server.views.baseurl + "/d79634e1-6063-4ec9-95ff-50acbf609ff5?includeUsageStatistics=true",
            text=response_xml,
        )
        view = server.views.get_by_id("d79634e1-6063-4ec9-95ff-50acbf609ff5", usage=True)

    assert "d79634e1-6063-4ec9-95ff-50acbf609ff5" == view.id
    assert "ENDANGERED SAFARI" == view.name
    assert "SafariSample/sheets/ENDANGEREDSAFARI" == view.content_url
    assert "3cc6cd06-89ce-4fdc-b935-5294135d6d42" == view.workbook_id
    assert "5de011f8-5aa9-4d5b-b991-f462c8dd6bb7" == view.owner_id
    assert "5241e88d-d384-4fd7-9c2f-648b5247efc5" == view.project_id
    assert {"tag1", "tag2"} == view.tags
    assert "2002-05-30T09:00:00Z" == format_datetime(view.created_at)
    assert "2002-06-05T08:00:59Z" == format_datetime(view.updated_at)
    assert "story" == view.sheet_type
    assert 7 == view.total_views


def test_get_by_id_missing_id(server: TSC.Server) -> None:
    with pytest.raises(TSC.MissingRequiredFieldError):
        server.views.get_by_id(None)


def test_get_with_usage(server: TSC.Server) -> None:
    response_xml = GET_XML_USAGE.read_text()
    with requests_mock.mock() as m:
        m.get(server.views.baseurl + "?includeUsageStatistics=true", text=response_xml)
        all_views, pagination_item = server.views.get(usage=True)

    assert "d79634e1-6063-4ec9-95ff-50acbf609ff5" == all_views[0].id
    assert 7 == all_views[0].total_views
    assert all_views[0].created_at is None
    assert all_views[0].updated_at is None
    assert all_views[0].sheet_type is None

    assert "fd252f73-593c-4c4e-8584-c032b8022adc" == all_views[1].id
    assert 13 == all_views[1].total_views
    assert "2002-05-30T09:00:00Z" == format_datetime(all_views[1].created_at)
    assert "2002-06-05T08:00:59Z" == format_datetime(all_views[1].updated_at)
    assert "story" == all_views[1].sheet_type


def test_get_with_usage_and_filter(server: TSC.Server) -> None:
    response_xml = GET_XML_USAGE.read_text()
    with requests_mock.mock() as m:
        m.get(server.views.baseurl + "?includeUsageStatistics=true&filter=name:in:[foo,bar]", text=response_xml)
        options = TSC.RequestOptions()
        options.filter.add(TSC.Filter(TSC.RequestOptions.Field.Name, TSC.RequestOptions.Operator.In, ["foo", "bar"]))
        all_views, pagination_item = server.views.get(req_options=options, usage=True)

    assert "ENDANGERED SAFARI" == all_views[0].name
    assert 7 == all_views[0].total_views
    assert "Overview" == all_views[1].name
    assert 13 == all_views[1].total_views


def test_get_before_signin(server: TSC.Server) -> None:
    server._auth_token = None
    with pytest.raises(TSC.NotSignedInError):
        server.views.get()


def test_populate_preview_image(server: TSC.Server) -> None:
    response = POPULATE_PREVIEW_IMAGE.read_bytes()
    with requests_mock.mock() as m:
        m.get(
            server.views.siteurl + "/workbooks/3cc6cd06-89ce-4fdc-b935-5294135d6d42/"
            "views/d79634e1-6063-4ec9-95ff-50acbf609ff5/previewImage",
            content=response,
        )
        single_view = TSC.ViewItem()
        single_view._id = "d79634e1-6063-4ec9-95ff-50acbf609ff5"
        single_view._workbook_id = "3cc6cd06-89ce-4fdc-b935-5294135d6d42"
        server.views.populate_preview_image(single_view)
        assert response == single_view.preview_image


def test_populate_preview_image_missing_id(server: TSC.Server) -> None:
    single_view = TSC.ViewItem()
    single_view._id = "d79634e1-6063-4ec9-95ff-50acbf609ff5"
    with pytest.raises(TSC.MissingRequiredFieldError):
        server.views.populate_preview_image(single_view)

    single_view._id = None
    single_view._workbook_id = "3cc6cd06-89ce-4fdc-b935-5294135d6d42"
    with pytest.raises(TSC.MissingRequiredFieldError):
        server.views.populate_preview_image(single_view)


def test_populate_image(server: TSC.Server) -> None:
    response = POPULATE_PREVIEW_IMAGE.read_bytes()
    with requests_mock.mock() as m:
        m.get(server.views.baseurl + "/d79634e1-6063-4ec9-95ff-50acbf609ff5/image", content=response)
        single_view = TSC.ViewItem()
        single_view._id = "d79634e1-6063-4ec9-95ff-50acbf609ff5"
        server.views.populate_image(single_view)
        assert response == single_view.image


def test_populate_image_unsupported(server: TSC.Server) -> None:
    server.version = "3.8"
    response = POPULATE_PREVIEW_IMAGE.read_bytes()
    with requests_mock.mock() as m:
        m.get(
            server.views.baseurl + "/d79634e1-6063-4ec9-95ff-50acbf609ff5/image?vizWidth=1920&vizHeight=1080",
            content=response,
        )
        single_view = TSC.ViewItem()
        single_view._id = "d79634e1-6063-4ec9-95ff-50acbf609ff5"

        req_option = TSC.ImageRequestOptions(viz_width=1920, viz_height=1080)

        with pytest.raises(UnsupportedAttributeError):
            server.views.populate_image(single_view, req_option)


def test_populate_image_viz_dimensions(server: TSC.Server) -> None:
    server.version = "3.23"
    response = POPULATE_PREVIEW_IMAGE.read_bytes()
    with requests_mock.mock() as m:
        m.get(
            server.views.baseurl + "/d79634e1-6063-4ec9-95ff-50acbf609ff5/image?vizWidth=1920&vizHeight=1080",
            content=response,
        )
        single_view = TSC.ViewItem()
        single_view._id = "d79634e1-6063-4ec9-95ff-50acbf609ff5"

        req_option = TSC.ImageRequestOptions(viz_width=1920, viz_height=1080)

        server.views.populate_image(single_view, req_option)
        assert response == single_view.image

        history = m.request_history


def test_populate_image_with_options(server: TSC.Server) -> None:
    response = POPULATE_PREVIEW_IMAGE.read_bytes()
    with requests_mock.mock() as m:
        m.get(
            server.views.baseurl + "/d79634e1-6063-4ec9-95ff-50acbf609ff5/image?resolution=high&maxAge=10",
            content=response,
        )
        single_view = TSC.ViewItem()
        single_view._id = "d79634e1-6063-4ec9-95ff-50acbf609ff5"
        req_option = TSC.ImageRequestOptions(imageresolution=TSC.ImageRequestOptions.Resolution.High, maxage=10)
        server.views.populate_image(single_view, req_option)
        assert response == single_view.image


def test_populate_pdf(server: TSC.Server) -> None:
    response = POPULATE_PDF.read_bytes()
    with requests_mock.mock() as m:
        m.get(
            server.views.baseurl
            + "/d79634e1-6063-4ec9-95ff-50acbf609ff5/pdf?type=letter&orientation=portrait&maxAge=5",
            content=response,
        )
        single_view = TSC.ViewItem()
        single_view._id = "d79634e1-6063-4ec9-95ff-50acbf609ff5"

        size = TSC.PDFRequestOptions.PageType.Letter
        orientation = TSC.PDFRequestOptions.Orientation.Portrait
        req_option = TSC.PDFRequestOptions(size, orientation, 5)

        server.views.populate_pdf(single_view, req_option)
        assert response == single_view.pdf


def test_populate_csv(server: TSC.Server) -> None:
    response = POPULATE_CSV.read_bytes()
    with requests_mock.mock() as m:
        m.get(server.views.baseurl + "/d79634e1-6063-4ec9-95ff-50acbf609ff5/data?maxAge=1", content=response)
        single_view = TSC.ViewItem()
        single_view._id = "d79634e1-6063-4ec9-95ff-50acbf609ff5"
        request_option = TSC.CSVRequestOptions(maxage=1)
        server.views.populate_csv(single_view, request_option)

        csv_file = b"".join(single_view.csv)
        assert response == csv_file


def test_populate_csv_default_maxage(server: TSC.Server) -> None:
    response = POPULATE_CSV.read_bytes()
    with requests_mock.mock() as m:
        m.get(server.views.baseurl + "/d79634e1-6063-4ec9-95ff-50acbf609ff5/data", content=response)
        single_view = TSC.ViewItem()
        single_view._id = "d79634e1-6063-4ec9-95ff-50acbf609ff5"
        server.views.populate_csv(single_view)

        csv_file = b"".join(single_view.csv)
        assert response == csv_file


def test_populate_image_missing_id(server: TSC.Server) -> None:
    single_view = TSC.ViewItem()
    single_view._id = None
    with pytest.raises(TSC.MissingRequiredFieldError):
        server.views.populate_image(single_view)


def test_populate_permissions(server: TSC.Server) -> None:
    response_xml = POPULATE_PERMISSIONS_XML.read_text()
    with requests_mock.mock() as m:
        m.get(server.views.baseurl + "/e490bec4-2652-4fda-8c4e-f087db6fa328/permissions", text=response_xml)
        single_view = TSC.ViewItem()
        single_view._id = "e490bec4-2652-4fda-8c4e-f087db6fa328"

        server.views.populate_permissions(single_view)
        permissions = single_view.permissions

        assert permissions[0].grantee.tag_name == "group"
        assert permissions[0].grantee.id == "c8f2773a-c83a-11e8-8c8f-33e6d787b506"
        assert permissions[0].capabilities == {
            TSC.Permission.Capability.ViewComments: TSC.Permission.Mode.Allow,
            TSC.Permission.Capability.Read: TSC.Permission.Mode.Allow,
            TSC.Permission.Capability.AddComment: TSC.Permission.Mode.Allow,
            TSC.Permission.Capability.ExportData: TSC.Permission.Mode.Allow,
            TSC.Permission.Capability.ExportImage: TSC.Permission.Mode.Allow,
        }


def test_add_permissions(server: TSC.Server) -> None:
    response_xml = UPDATE_PERMISSIONS.read_text()

    single_view = TSC.ViewItem()
    single_view._id = "21778de4-b7b9-44bc-a599-1506a2639ace"

    bob = UserItem.as_reference("7c37ee24-c4b1-42b6-a154-eaeab7ee330a")
    group_of_people = GroupItem.as_reference("5e5e1978-71fa-11e4-87dd-7382f5c437af")

    new_permissions = [PermissionsRule(bob, {"Write": "Allow"}), PermissionsRule(group_of_people, {"Read": "Deny"})]

    with requests_mock.mock() as m:
        m.put(server.views.baseurl + "/21778de4-b7b9-44bc-a599-1506a2639ace/permissions", text=response_xml)
        permissions = server.views.update_permissions(single_view, new_permissions)

    assert permissions[0].grantee.tag_name == "group"
    assert permissions[0].grantee.id == "5e5e1978-71fa-11e4-87dd-7382f5c437af"
    assert permissions[0].capabilities == {TSC.Permission.Capability.Read: TSC.Permission.Mode.Deny}

    assert permissions[1].grantee.tag_name == "user"
    assert permissions[1].grantee.id == "7c37ee24-c4b1-42b6-a154-eaeab7ee330a"
    assert permissions[1].capabilities == {TSC.Permission.Capability.Write: TSC.Permission.Mode.Allow}


def test_update_tags(server: TSC.Server) -> None:
    add_tags_xml = ADD_TAGS_XML.read_text()
    update_xml = UPDATE_XML.read_text()
    with requests_mock.mock() as m:
        m.put(server.views.baseurl + "/d79634e1-6063-4ec9-95ff-50acbf609ff5/tags", text=add_tags_xml)
        m.delete(server.views.baseurl + "/d79634e1-6063-4ec9-95ff-50acbf609ff5/tags/b", status_code=204)
        m.delete(server.views.baseurl + "/d79634e1-6063-4ec9-95ff-50acbf609ff5/tags/d", status_code=204)
        m.put(server.views.baseurl + "/d79634e1-6063-4ec9-95ff-50acbf609ff5", text=update_xml)
        single_view = TSC.ViewItem()
        single_view._id = "d79634e1-6063-4ec9-95ff-50acbf609ff5"
        single_view._initial_tags.update(["a", "b", "c", "d"])
        single_view.tags.update(["a", "c", "e"])
        updated_view = server.views.update(single_view)

    assert single_view.tags == updated_view.tags
    assert single_view._initial_tags == updated_view._initial_tags


def test_populate_excel(server: TSC.Server) -> None:
    server.version = "3.8"
    response = POPULATE_EXCEL.read_bytes()
    with requests_mock.mock() as m:
        m.get(server.views.baseurl + "/d79634e1-6063-4ec9-95ff-50acbf609ff5/crosstab/excel?maxAge=1", content=response)
        single_view = TSC.ViewItem()
        single_view._id = "d79634e1-6063-4ec9-95ff-50acbf609ff5"
        request_option = TSC.ExcelRequestOptions(maxage=1)
        server.views.populate_excel(single_view, request_option)

        excel_file = b"".join(single_view.excel)
        assert response == excel_file


def test_filter_excel(server: TSC.Server) -> None:
    server.version = "3.8"
    response = POPULATE_EXCEL.read_bytes()
    with requests_mock.mock() as m:
        m.get(server.views.baseurl + "/d79634e1-6063-4ec9-95ff-50acbf609ff5/crosstab/excel?maxAge=1", content=response)
        single_view = TSC.ViewItem()
        single_view._id = "d79634e1-6063-4ec9-95ff-50acbf609ff5"
        request_option = TSC.ExcelRequestOptions(maxage=1)
        request_option.vf("stuff", "1")
        server.views.populate_excel(single_view, request_option)

        excel_file = b"".join(single_view.excel)
        assert response == excel_file


def test_pdf_height(server: TSC.Server) -> None:
    server.version = "3.8"
    response = POPULATE_PDF.read_bytes()
    with requests_mock.mock() as m:
        m.get(
            server.views.baseurl + "/d79634e1-6063-4ec9-95ff-50acbf609ff5/pdf?vizHeight=1080&vizWidth=1920",
            content=response,
        )
        single_view = TSC.ViewItem()
        single_view._id = "d79634e1-6063-4ec9-95ff-50acbf609ff5"

        req_option = TSC.PDFRequestOptions(
            viz_height=1080,
            viz_width=1920,
        )

        server.views.populate_pdf(single_view, req_option)
        assert response == single_view.pdf


def test_pdf_errors(server: TSC.Server) -> None:
    req_option = TSC.PDFRequestOptions(viz_height=1080)
    with pytest.raises(ValueError):
        req_option.get_query_params()
    req_option = TSC.PDFRequestOptions(viz_width=1920)
    with pytest.raises(ValueError):
        req_option.get_query_params()


def test_view_get_all_fields(server: TSC.Server) -> None:
    server.version = "3.21"
    response_xml = GET_XML_ALL_FIELDS.read_text()

    ro = TSC.RequestOptions()
    ro.all_fields = True

    with requests_mock.mock() as m:
        m.get(f"{server.views.baseurl}?fields=_all_", text=response_xml)
        views, _ = server.views.get(req_options=ro)

    assert views[0].id == "2bdcd787-dcc6-4a5d-bc61-2846f1ef4534"
    assert views[0].name == "Overview"
    assert views[0].content_url == "Superstore/sheets/Overview"
    assert views[0].created_at == parse_datetime("2024-02-14T04:42:09Z")
    assert views[0].updated_at == parse_datetime("2024-02-14T04:42:09Z")
    assert views[0].sheet_type == "dashboard"
    assert views[0].favorites_total == 0
    assert views[0].view_url_name == "Overview"
    assert isinstance(views[0].workbook, TSC.WorkbookItem)
    assert views[0].workbook.id == "9df3e2d1-070e-497a-9578-8cc557ced9df"
    assert views[0].workbook.name == "Superstore"
    assert views[0].workbook.content_url == "Superstore"
    assert views[0].workbook.show_tabs
    assert views[0].workbook.size == 2
    assert views[0].workbook.created_at == parse_datetime("2024-02-14T04:42:09Z")
    assert views[0].workbook.updated_at == parse_datetime("2024-02-14T04:42:10Z")
    assert views[0].workbook.sheet_count == 9
    assert not views[0].workbook.has_extracts
    assert isinstance(views[0].owner, TSC.UserItem)
    assert views[0].owner.email == "bob@example.com"
    assert views[0].owner.fullname == "Bob"
    assert views[0].owner.id == "ee8bc9ca-77fe-4ae0-8093-cf77f0ee67a9"
    assert views[0].owner.last_login == parse_datetime("2025-02-04T06:39:20Z")
    assert views[0].owner.name == "bob@example.com"
    assert views[0].owner.site_role == "SiteAdministratorCreator"
    assert isinstance(views[0].project, TSC.ProjectItem)
    assert views[0].project.id == "669ca36b-492e-4ccf-bca1-3614fe6a9d7a"
    assert views[0].project.name == "Samples"
    assert views[0].project.description == "This project includes automatically uploaded samples."
    assert views[0].total_views == 0
    assert isinstance(views[0].location, TSC.LocationItem)
    assert views[0].location.id == "669ca36b-492e-4ccf-bca1-3614fe6a9d7a"
    assert views[0].location.type == "Project"
    assert views[1].id == "2a3fd19d-9129-413d-9ff7-9dfc36bf7f7e"
    assert views[1].name == "Product"
    assert views[1].content_url == "Superstore/sheets/Product"
    assert views[1].created_at == parse_datetime("2024-02-14T04:42:09Z")
    assert views[1].updated_at == parse_datetime("2024-02-14T04:42:09Z")
    assert views[1].sheet_type == "dashboard"
    assert views[1].favorites_total == 0
    assert views[1].view_url_name == "Product"
    assert isinstance(views[1].workbook, TSC.WorkbookItem)
    assert views[1].workbook.id == "9df3e2d1-070e-497a-9578-8cc557ced9df"
    assert views[1].workbook.name == "Superstore"
    assert views[1].workbook.content_url == "Superstore"
    assert views[1].workbook.show_tabs
    assert views[1].workbook.size == 2
    assert views[1].workbook.created_at == parse_datetime("2024-02-14T04:42:09Z")
    assert views[1].workbook.updated_at == parse_datetime("2024-02-14T04:42:10Z")
    assert views[1].workbook.sheet_count == 9
    assert not views[1].workbook.has_extracts
    assert isinstance(views[1].owner, TSC.UserItem)
    assert views[1].owner.email == "bob@example.com"
    assert views[1].owner.fullname == "Bob"
    assert views[1].owner.id == "ee8bc9ca-77fe-4ae0-8093-cf77f0ee67a9"
    assert views[1].owner.last_login == parse_datetime("2025-02-04T06:39:20Z")
    assert views[1].owner.name == "bob@example.com"
    assert views[1].owner.site_role == "SiteAdministratorCreator"
    assert isinstance(views[1].project, TSC.ProjectItem)
    assert views[1].project.id == "669ca36b-492e-4ccf-bca1-3614fe6a9d7a"
    assert views[1].project.name == "Samples"
    assert views[1].project.description == "This project includes automatically uploaded samples."
    assert views[1].total_views == 0
    assert isinstance(views[1].location, TSC.LocationItem)
    assert views[1].location.id == "669ca36b-492e-4ccf-bca1-3614fe6a9d7a"
    assert views[1].location.type == "Project"
    assert views[2].id == "459eda9a-85e4-46bf-a2f2-62936bd2e99a"
    assert views[2].name == "Customers"
    assert views[2].content_url == "Superstore/sheets/Customers"
    assert views[2].created_at == parse_datetime("2024-02-14T04:42:09Z")
    assert views[2].updated_at == parse_datetime("2024-02-14T04:42:09Z")
    assert views[2].sheet_type == "dashboard"
    assert views[2].favorites_total == 0
    assert views[2].view_url_name == "Customers"
    assert isinstance(views[2].workbook, TSC.WorkbookItem)
    assert views[2].workbook.id == "9df3e2d1-070e-497a-9578-8cc557ced9df"
    assert views[2].workbook.name == "Superstore"
    assert views[2].workbook.content_url == "Superstore"
    assert views[2].workbook.show_tabs
    assert views[2].workbook.size == 2
    assert views[2].workbook.created_at == parse_datetime("2024-02-14T04:42:09Z")
    assert views[2].workbook.updated_at == parse_datetime("2024-02-14T04:42:10Z")
    assert views[2].workbook.sheet_count == 9
    assert not views[2].workbook.has_extracts
    assert isinstance(views[2].owner, TSC.UserItem)
    assert views[2].owner.email == "bob@example.com"
    assert views[2].owner.fullname == "Bob"
    assert views[2].owner.id == "ee8bc9ca-77fe-4ae0-8093-cf77f0ee67a9"
    assert views[2].owner.last_login == parse_datetime("2025-02-04T06:39:20Z")
    assert views[2].owner.name == "bob@example.com"
    assert views[2].owner.site_role == "SiteAdministratorCreator"
    assert isinstance(views[2].project, TSC.ProjectItem)
    assert views[2].project.id == "669ca36b-492e-4ccf-bca1-3614fe6a9d7a"
    assert views[2].project.name == "Samples"
    assert views[2].project.description == "This project includes automatically uploaded samples."
    assert views[2].total_views == 0
    assert isinstance(views[2].location, TSC.LocationItem)
    assert views[2].location.id == "669ca36b-492e-4ccf-bca1-3614fe6a9d7a"
    assert views[2].location.type == "Project"


def make_view() -> TSC.ViewItem:
    view = TSC.ViewItem()
    view._id = "1234"
    return view


@pytest.mark.parametrize("view", [make_view, "1234"])
def test_delete_view(server: TSC.Server, view: TSC.ViewItem | str) -> None:
    server.version = "3.27"
    id_ = getattr(view, "id", view)
    with requests_mock.mock() as m:
        m.delete(f"{server.views.baseurl}/{id_}")
        server.views.delete(view)
        assert m.called
        assert m.call_count == 1
