<!--
  - SPDX-FileCopyrightText: 2020-2024 Nextcloud GmbH and Nextcloud contributors
  - SPDX-License-Identifier: AGPL-3.0-or-later
-->
# API version 1

In this document, the Notes API major version 1 and all its minor versions are described. An introduction with general information about versions, capabilities, compatibility between versions, authentication and input parameters can be found in the [README](README.md).


## Minor versions

| API version | Introduced with app version | Remarkable Changes                                 |
|:-----------:|:----------------------------|:---------------------------------------------------|
|   **1.0**   | Notes 3.3 (May 2020)        | Separate title, no auto rename based on content    |
|   **1.1**   | Notes 3.4 (May 2020)        | Filter "Get all notes" by category                 |
|   **1.2**   | Notes 4.1 (June 2021)       | Preventing lost updates, read-only notes, settings |
|   **1.3**   | Notes 4.5 (August 2022)     | Allow custom file suffixes                         |
|   **1.4**   | Notes 4.9 (August 2025)     | Add external image api                             |



## Note attributes

The app and the API is mainly about notes. So, let's have a look about the attributes of a note. The description of endpoints and operations will refer to this attribute definition.

| Attribute | Type | Description | since API version |
|:----------|:-----|:------------|:------------------|
| `id` | integer (read‑only) | Every note has a unique identifier which is created by the server. It can be used to query and update a specific note. | 1.0 |
| `etag` | string (read‑only) | The note's entity tag (ETag) indicates if a note's attribute has changed. I.e., if the note changes, the ETag changes, too. Clients can use the ETag for detecting if the local note has to be updated from server and for optimistic concurrency control (see section [Preventing lost updates and conflict solution](#preventing-lost-updates-and-conflict-solution)). | 1.2 |
| `readonly` | boolean (read‑only) | Indicates if the note is read-only. This is `true`, e.g., if a file or folder was shared by another user without allowing editing. If this attribute is `true`, then all read/write attributes become read-only; except for the `favorite` attribute. | 1.2 |
| `content` | string (read/write) | Notes can contain arbitrary text. Formatting should be done using Markdown, but not every markup can be supported by every client. Therefore, markup should be used with care. | 1.0 |
| `title` | string (read/write) | The note's title is also used as filename for the note's file. Therefore, some special characters are automatically removed and a sequential number is added if a note with the same title in the same category exists. When saving a title, the sanitized value is returned and should be adopted by your client. | 1.0 |
| `category` | string (read/write) | Every note is assigned to a category. By default, the category is an empty string (not null), which means the note is uncategorized. Categories are mapped to folders in the file backend. Illegal characters are automatically removed and the respective folder is automatically created. Sub-categories (mapped to sub-folders) can be created by using `/` as delimiter. | 1.0 |
| `favorite` | boolean (read/write) | If a note is marked as favorite, it is displayed at the top of the notes' list. Default is `false`. | 1.0 |
| `modified` | integer (read/write) | Unix timestamp for the last modified date/time of the note. If not provided on note creation or content update, the current time is used. | 1.0 |


## Settings

Since API version 1.2, it is possible to change app settings using the API. The following settings attributes exist:

| Attribute | Type | Description | since API version |
|:----------|:-----|:------------|:------------------|
| `notesPath` | string | Path to the folder, where note's files are stored in Nextcloud. The path must be relative to the user folder. Default is the localized string `Notes`. | 1.2 |
| `fileSuffix` | string | Newly created note's files will have this file suffix. For API version 1.2, only the values `.txt` or `.md` are allowed. Since API version 1.3, also custom suffixes can be chosen. Default is `.txt`. | 1.2 |


## Endpoints and Operations

The base URL for all calls is:

    https://user:password@yournextcloud.com/index.php/apps/notes/api/v1/

All defined routes in the specification are appended to this url. To access all notes for instance use this url (here shown as `curl` command):

    curl -u user:password -H "Accept: application/json" https://yournextcloud.com/index.php/apps/notes/api/v1/notes



### Get all notes (`GET /notes`)
<details><summary>Details</summary>

#### Request parameters
| Parameter | Type | Description | since API version |
|:----------|:-----|:------------|:------------------|
| `category` | string, optional | Filter the result by category name, e.g. `?category=recipes`. Notes with another category are not included in the result. *Compatibility note:* before API v1.1, this parameter is ignored; i.e., the result contains all notes regardless of this parameter. | 1.1 |
| `exclude` | string, optional | Fields which should be excluded from response, seperated with a comma e.g.: `?exclude=content,title`. You can use this in order to reduce transferred data size if you are interested in specific attributes, only. | 1.0 |
| `pruneBefore` | integer, optional | All notes without change before of this Unix timestamp are purged from the response, i.e. only the attribute `id` is included. You should use the Unix timestamp value from the last request's HTTP response header `Last-Modified` in order to reduce transferred data size. | 1.0 |
| `chunkSize` | integer, optional | The response will contain no more than the given number of full notes. If there are more notes, then the result is chunked and the HTTP response header `X-Notes-Chunk-Cursor` is sent with a string value. In order to request the next chunk, a new request have to be made with parameter `chunkCursor` filled with that string value. *Compatibility note:* before API v1.2, this parameter is ignored; i.e., the result contains all notes regardless of this parameter. | 1.2 |
| `chunkCursor` | string, optional | To be used together with the parameter `chunkSize`. You must use the string value from the last request's HTTP response header `X-Notes-Chunk-Cursor` in order to get the next chunk of notes. Don't use this parameter for requesting the first chunk. *Compatibility note:* before API v1.2, this parameter is ignored; i.e., the result contains all notes regardless of this parameter. | 1.2 |
| `If-None-Match` | HTTP header, optional | Use this in order to reduce transferred data size (see [HTTP ETag](https://en.wikipedia.org/wiki/HTTP_ETag)). You should use the value from the last request's HTTP response header `ETag`. | 1.0 |

#### Response
##### 200 OK
- **HTTP Header**:
	- `ETag` (see [HTTP ETag](https://en.wikipedia.org/wiki/HTTP_ETag)).
	- `X-Notes-Chunk-Cursor`: Only if `chunkSize` is provided and not `0` and if the response does not contain all remaining notes. In this case, the response does not contain pruned notes. In order to get the next chunk, you will have to make a new request and use this header value as request parameter `chunkCursor`. The last chunk response will not contain this header but it will contain all pruned notes. In summary: a client have to repeatedly request the notes list from server with the desired `chunkSize` and with updated `chunkCursor` until the response does not contain any `X-Notes-Chunk-Cursor` HTTP header – only this last request can be used to check for deleted notes.
	- `X-Notes-Chunk-Pending`: number of pending notes that have to be requested using the chunk cursor provided in the HTTP response header `X-Notes-Chunk-Cursor`.
- **Body**: list of notes (see section [Note attributes](#note-attributes)), example:
```js
[
    {
        "id": 76,
        "etag": "be284e00488c61c101ee28309d235e0b",
        "readonly": false,
        "modified": 1376753464,
        "title": "New note",
        "category": "sub-directory",
        "content": "New note\n and something more",
        "favorite": false
    }, // etc
]
```

##### 401 Unauthorized
No valid authentication credentials supplied.
</details>


### Get single note (`GET /notes/{id}`)
<details><summary>Details</summary>

#### Request parameters
| Parameter | Type | Description |
|:------|:-----|:-----|
| `id` | integer, required (path) | ID of the note to query. |
| `If-None-Match` | HTTP header, optional | Use this in order to reduce transferred data size (see [HTTP ETag](https://en.wikipedia.org/wiki/HTTP_ETag)). You should use the value from the note's attribute `etag` or from the last request's HTTP response header `ETag`. | 1.2 |

#### Response
##### 200 OK
- **HTTP Header**: `ETag` (see [HTTP ETag](https://en.wikipedia.org/wiki/HTTP_ETag)). The value is identical to the note's attribute `etag` (see section [Note attributes](#note-attributes)).
- **Body**: note (see section [Note attributes](#note-attributes)), example:
```js
{
    "id": 76,
    "etag": "be284e00488c61c101ee28309d235e0b",
    "readonly": false,
    "modified": 1376753464,
    "title": "New note",
    "category": "sub-directory",
    "content": "New note\n and something more",
    "favorite": false
}
```
##### 400 Bad Request
Invalid ID supplied.

##### 401 Unauthorized
No valid authentication credentials supplied.

##### 404 Not Found
Note not found.
</details>


### Create note (`POST /notes`)
<details><summary>Details</summary>

#### Request parameters
- **Body**: some or all "read/write" attributes (see section [Note attributes](#note-attributes)), example:
```js
{
    "title": "New note",
    "category": "Category/Sub Category",
    "content": "New note\n and something more",
}
```

#### Response
##### 200 OK
- **Body**: note (see section [Note attributes](#note-attributes)), example see section [Get single note](#get-single-note-get-notesid).

##### 400 Bad Request
Invalid ID supplied.

##### 401 Unauthorized
No valid authentication credentials supplied.

##### 507 Insufficient Storage
Not enough free storage for saving the note's content.
</details>


### Update note (`PUT /notes/{id}`)
<details><summary>Details</summary>

#### Request parameters
| Parameter | Type | Description |
|:------|:-----|:-----|
| `id` | integer, required (path) | ID of the note to update. |
| `If-Match` | HTTP header, optional | Use this for optimistic concurrency control (optional, but strongly recommended in order to prevent lost updates). As value of this HTTP header, the client has to use the last known note's etag (see section [Note attributes](#note-attributes)). If the note has changed in the meanwhile (concurrent change), the update request is blocked with HTTP status 412 (see below). Otherwise, the request will be processed normally. | 1.2 |
- **Body**: some or all "read/write" attributes (see section [Note attributes](#note-attributes)), example see section [Create note](#create-note-post-notes).

#### Response
##### 200 OK
- **Body**: note (see section [Note attributes](#note-attributes)), example see section [Get single note](#get-single-note-get-notesid).

##### 400 Bad Request
Invalid ID supplied.

##### 401 Unauthorized
No valid authentication credentials supplied.

##### 403 Forbidden
The note is read-only.

##### 404 Not Found
Note not found.

##### 412 Precondition Failed
*(since API v1.2)*
Update cannot be performed since the note has been changed on the server in the meanwhile (concurrent change). The body contains the current note's state from server (see section [Note attributes](#note-attributes)), example see section [Get single note](#get-single-note-get-notesid). The client should use this response data in order to perform a conflict solution (see section [Preventing lost updates and conflict solution](#preventing-lost-updates-and-conflict-solution)).

##### 507 Insufficient Storage
Not enough free storage for saving the note's content.
</details>


### Delete note (`DELETE /notes/{id}`)
<details><summary>Details</summary>

#### Request parameters
| Parameter | Type | Description |
|:------|:-----|:-----|
| `id` | integer, required (path) | ID of the note to delete. |

#### Response
##### 200 OK
Note is deleted.

##### 400 Bad Request
Invalid ID supplied.

##### 401 Unauthorized
No valid authentication credentials supplied.

##### 403 Forbidden
The note is read-only.

##### 404 Not Found
Note not found.
</details>


### Get settings (`GET /settings`)
<details><summary>Details</summary>

*(since API v1.2)*

#### Request parameters
None.

#### Response
##### 200 OK
- **Body**: user's app settings (see section [Settings](#settings)), example:
```js
{
    "notesPath": "Notes",
    "fileSuffix": ".txt"
}
```

##### 400 Bad Request
Endpoint not supported by installed notes app version (requires API version 1.2).

##### 401 Unauthorized
No valid authentication credentials supplied.
</details>


### Change settings (`PUT /settings`)
<details><summary>Details</summary>

*(since API v1.2)*

#### Request parameters
- **Body**: some or all settings attributes (see section [Settings](#settings)).
Omitted settings attributes are not changed.
Empty values are replaced by the settings attribute's default value.
All values are sanitized (e.g. prevent path traversal attacks, check allowed suffixes), so the result can differ from the request (the request will still succeed).
The client may show an information to the user if the response differs from the request.
Example:
```js
{
    "fileSuffix": ".md"
}
```

#### Response
##### 200 OK
- **Body**: user's app settings after validation (see section [Settings](#settings)), example see section [Get settings](#get-settings-get-settings).

##### 400 Bad Request
Endpoint not supported by installed notes app version (requires API version 1.2).

##### 401 Unauthorized
No valid authentication credentials supplied.
</details>



### Get attachment (`GET /attachment/{id}`)
<details><summary>Details</summary>

*(since API v1.4)*

#### Request parameters
| Parameter | Type                         | Description                                |
|:----------|:-----------------------------|:-------------------------------------------|
| `id`      | integer, required (path)     | ID of the note to load the attachment from |
| `path`    | string, required (request)   | Path or name of the attachment to load.    |

Example:

```bash
curl -u "user:password" "https://yournextcloud.com/index.php/apps/notes/api/v1.4/attachment/<id>?path=<path>" -o <outputfilename>.jpg
```


#### Response
##### 200 OK
- **Body**: Image or File

##### 400 Bad Request
Endpoint not supported by installed notes app version (requires API version 1.4).

##### 401 Unauthorized
No valid authentication credentials supplied.
</details>


### Put attachment (`POST /attachment/{id}`)
<details><summary>Details</summary>

*(since API v1.4)*

#### Request parameters
| Parameter | Type                    | Description                                     |
|:----------|:------------------------|:------------------------------------------------|
| `id`      | integer, required (path)| ID of the note to upload the attachment to      |

Example:

```bash
curl -u "user:password" \
       -X POST \
       -F "file=@/path/to/image.png" \
       "https://yournextcloud.com/index.php/apps/notes/api/v1.4/attachment/<id>"

# The post request will return the filename that was generated:
{"filename":"d8aef2005b4f815fec8ade5388240f2c.png"}
```

#### Response
##### 200 OK
- **Body**: Filename in json encoded:
```js
{
    "filename": "image.jpg"
}
```

##### 400 Bad Request
Endpoint not supported by installed notes app version (requires API version 1.4).

##### 401 Unauthorized
No valid authentication credentials supplied.
</details>

## Preventing lost updates and conflict solution

While changing a note using a Notes client, the same note may be changed by another client.
In order to prevent lost updates of those concurrent changes, the notes API uses a well established mechanism called [optimistic concurrency control](https://en.wikipedia.org/wiki/Optimistic_concurrency_control).
For this purpose, notes have the attribute `etag` which is an identifier that changes if (and only if) the note changes on the server.
Clients have to store the `etag` for every note and send its value with every update request (HTTP header `If-Match`, see section [Update note](#update-note-put-notesid)).
If there was no parallel change on the server (i.e., the `etag` on server is the same as the one send from the client), the update request is performed as usual.
But if there was a parallel change, the `etag` on the server has changed and the server will refuse the update request.

In this case, the client has to perform a conflict resolution, i.e. the local changes have to be merged with the remote changes.
In order to compare local changes with remote changes, it is useful that the client stores the full note's state as reference state before performing any local updates.
If an update conflict occurs, the client can use this reference state in order to merge all changes attribute-wise:
- Attributes, that have changed only locally or remotely, can be merged by picking the (local resp. remote) change.
- Attributes, that have changed both localy and remotely, have to be merged (see below).

There are several options on how to merge an attribute:
- a) *Let the user decide*: ask the user whether i) overwrite local changes, ii) overwrite remote changes, or iii) save local (or remote) changes as new note.
- b) *Let the user merge*: provide an interface which allows for merging the files (you know it from your version control).
- c) *Try to merge automatically*: merge all changes automatically, e.g. for the `content` attribute using the [google-diff-match-patch](https://code.google.com/p/google-diff-match-patch/) ([Demo](https://neil.fraser.name/software/diff_match_patch/svn/trunk/demos/demo_patch.html), [Code](https://github.com/bystep15/google-diff-match-patch)) library.

