#include "Application.hpp"
#include "MAS_CocoaAppUtils.hpp"
#include "MAS_MacOSAppUtils.hpp"
#include "MAS_NSImageLoader.hpp"
#include "MAS_NSViewContext.hpp"
#include "MAS_NSViewOffscreenContext.hpp"
#include "MAS_ParameterDialog.hpp"
#include "MAS_IncludeCocoaHeaders.hpp"

#include "NUIE_NodeTree.hpp"
#include "NUIE_FeatureToggle.hpp"
#include "BI_BuiltInNodes.hpp"

static const std::vector<NSString*> IconResourceNames = {
	@"Boolean.png",
	@"Integer.png",
	@"Double.png",
	@"IntegerIncremented.png",
	@"DoubleIncremented.png",
	@"DoubleDistributed.png",
	@"Addition.png",
	@"Subtraction.png",
	@"Multiplication.png",
	@"Division.png",
	@"Floor.png",
	@"Ceil.png",
	@"Abs.png",
	@"Negative.png",
	@"Sqrt.png",
	@"ListBuilder.png",
	@"Viewer.png"
};

class AppImageLoader : public MAS::NSImageLoader
{
public:
	AppImageLoader () :
	MAS::NSImageLoader ()
	{
		
	}
	
private:
	virtual NSImage* CreateNSImage (const NUIE::IconId& iconId) override
	{
		return [NSImage imageNamed : IconResourceNames[iconId.GetId () - 1]];
	}
};

enum class ContextType
{
	NSView,
	NSViewOffscreen
};

static NUIE::NativeDrawingContextPtr CreateDrawingContext (ContextType contextType)
{
	if (contextType == ContextType::NSView) {
		MAS::NSImageLoaderPtr imageLoader (new AppImageLoader ());
		return NUIE::NativeDrawingContextPtr (new MAS::NSViewContext (imageLoader));
	} else if (contextType == ContextType::NSViewOffscreen) {
		MAS::NSViewOffscreenContext::Orientation orientation = MAS::NSViewOffscreenContext::Orientation::Normal;
		MAS::NSImageLoaderPtr imageLoader (new AppImageLoader ());
		return NUIE::NativeDrawingContextPtr (new MAS::NSViewOffscreenContext (orientation, imageLoader));
	}
	DBGBREAK ();
	return nullptr;
}

@interface WindowController : NSWindowController<NSWindowDelegate>
{
@private AppNodeUIEnvironment* uiEnvironment;
}
@end

@implementation WindowController

- (id) init
{
	self = [super init];
	if (self) {
		uiEnvironment = nil;
	}
	return self;
}

- (void) setUIEnvironment : (AppNodeUIEnvironment*) newUIEnvironment
{
	uiEnvironment = newUIEnvironment;
}

- (void) windowDidLoad
{
    
}

- (void) windowWillClose : (NSNotification*) notification
{
	#pragma unused (notification)
	[NSApp terminate : self];
}

- (void) windowDidResize : (NSNotification *) notification
{
	#pragma unused (notification)
	NSRect clientRect = [[[self window] contentView] frame];
	uiEnvironment->Resize (0, 0, clientRect.size.width, clientRect.size.height);
}

@end

static const NUIE::BasicSkinParams& GetAppSkinParams ()
{
	static const NUIE::BasicSkinParams skinParams (
		/*backgroundColor*/ NUIE::Color (250, 250, 250),
		/*connectionLinePen*/ NUIE::Pen (NUIE::Color (38, 50, 56), 1.0),
		/*connectionMarker*/ NUIE::SkinParams::ConnectionMarker::Circle,
		/*connectionMarkerSize*/ NUIE::Size (8.0, 8.0),
		/*nodePadding*/ 5.0,
		/*nodeBorderPen*/ NUIE::Pen (NUIE::Color (38, 50, 56), 1.0),
		/*nodeHeaderTextFont*/ NUIE::Font (L"Helvetica", 16.0),
		/*nodeHeaderTextColor*/ NUIE::Color (250, 250, 250),
		/*nodeHeaderErrorTextColor*/ NUIE::Color (250, 250, 250),
		/*nodeHeaderBackgroundColor*/ NUIE::Color (41, 127, 255),
		/*nodeHeaderErrorBackgroundColor*/ NUIE::Color (199, 80, 80),
		/*nodeContentTextFont*/ NUIE::Font (L"Helvetica", 14.0),
		/*nodeContentTextColor*/ NUIE::Color (0, 0, 0),
		/*nodeContentBackgroundColor*/ NUIE::Color (236, 236, 236),
		/*slotTextColor*/ NUIE::Color (0, 0, 0),
		/*slotTextBackgroundColor*/ NUIE::Color (246, 246, 246),
		/*slotMarker*/ NUIE::SkinParams::SlotMarker::Circle,
		/*hiddenSlotMarker*/ NUIE::SkinParams::HiddenSlotMarker::Arrow,
		/*slotMarkerSize*/ NUIE::Size (8.0, 8.0),
		/*selectionBlendColor*/ NUIE::BlendColor (NUIE::Color (41, 127, 255), 0.25),
		/*disabledBlendColor*/ NUIE::BlendColor (NUIE::Color (0, 138, 184), 0.2),
		/*selectionRectPen*/ NUIE::Pen (NUIE::Color (41, 127, 255), 1.0),
		/*nodeSelectionRectPen*/ NUIE::Pen (NUIE::Color (41, 127, 255), 3.0),
		/*buttonBorderPen*/ NUIE::Pen (NUIE::Color (146, 152, 155), 1.0),
		/*buttonBackgroundColor*/ NUIE::Color (217, 217, 217),
		/*textPanelTextColor*/ NUIE::Color (0, 0, 0),
		/*textPanelBackgroundColor*/ NUIE::Color (236, 236, 236),
		/*groupNameFont*/ NUIE::Font (L"Helvetica", 16.0),
		/*groupNameColor*/ NUIE::Color (0, 0, 0),
		/*groupBackgroundColors*/ NUIE::NamedColorSet ({
			{ NE::LocalizeString (L"Blue"), NUIE::Color (160, 200, 240) },
			{ NE::LocalizeString (L"Green"), NUIE::Color (160, 239, 160) },
			{ NE::LocalizeString (L"Red"), NUIE::Color (239, 189, 160) }
		}),
		/*groupPadding*/ 12.0
	);
	return skinParams;
}

static void AddNodeTreeItem (NUIE::NodeTree& nodeTree, size_t groupIndex, const std::wstring& name, const NUIE::IconId& iconId, const NUIE::CreatorFunction& creator)
{
	nodeTree.AddItem (groupIndex, name, [=] (const NUIE::Point& position) {
		NUIE::UINodePtr node = creator (position);
		BI::BasicUINodePtr basicNode = std::dynamic_pointer_cast<BI::BasicUINode> (node);
		if (basicNode != nullptr) {
			basicNode->SetIconId (iconId);
		}
		return node;
	});
}

static void InitNodeTree (NUIE::NodeTree& nodeTree)
{
	size_t inputNodes = nodeTree.AddGroup (L"Input Nodes");
	AddNodeTreeItem (nodeTree, inputNodes, L"Boolean", NUIE::IconId (1), [&] (const NUIE::Point& position) {
		return NUIE::UINodePtr (new BI::BooleanNode (NE::LocString (L"Boolean"), position, true));
	});
	AddNodeTreeItem (nodeTree, inputNodes, L"Integer", NUIE::IconId (2), [&] (const NUIE::Point& position) {
		return NUIE::UINodePtr (new BI::IntegerUpDownNode (NE::LocString (L"Integer"), position, 0, 1));
	});
	AddNodeTreeItem (nodeTree, inputNodes, L"Number", NUIE::IconId (3), [&] (const NUIE::Point& position) {
		return NUIE::UINodePtr (new BI::DoubleUpDownNode (NE::LocString (L"Number"), position, 0.0, 1.0));
	});
	AddNodeTreeItem (nodeTree, inputNodes, L"Integer Increment", NUIE::IconId (4), [&] (const NUIE::Point& position) {
		return NUIE::UINodePtr (new BI::IntegerIncrementedNode (NE::LocString (L"Integer Increment"), position));
	});
	AddNodeTreeItem (nodeTree, inputNodes, L"Number Increment", NUIE::IconId (5), [&] (const NUIE::Point& position) {
		return NUIE::UINodePtr (new BI::DoubleIncrementedNode (NE::LocString (L"Number Increment"), position));
	});
	AddNodeTreeItem (nodeTree, inputNodes, L"Number Distribution", NUIE::IconId (6), [&] (const NUIE::Point& position) {
		return NUIE::UINodePtr (new BI::DoubleDistributedNode (NE::LocString (L"Number Distribution"), position));
	});
	size_t arithmeticNodes = nodeTree.AddGroup (L"Mathematical Nodes");
	AddNodeTreeItem (nodeTree, arithmeticNodes, L"Addition", NUIE::IconId (7), [&] (const NUIE::Point& position) {
		return NUIE::UINodePtr (new BI::AdditionNode (NE::LocString (L"Addition"), position));
	});
	AddNodeTreeItem (nodeTree, arithmeticNodes, L"Subtraction", NUIE::IconId (8), [&] (const NUIE::Point& position) {
		return NUIE::UINodePtr (new BI::SubtractionNode (NE::LocString (L"Subtraction"), position));
	});
	AddNodeTreeItem (nodeTree, arithmeticNodes, L"Multiplication", NUIE::IconId (9), [&] (const NUIE::Point& position) {
		return NUIE::UINodePtr (new BI::MultiplicationNode (NE::LocString (L"Multiplication"), position));
	});
	AddNodeTreeItem (nodeTree, arithmeticNodes, L"Division", NUIE::IconId (10), [&] (const NUIE::Point& position) {
		return NUIE::UINodePtr (new BI::DivisionNode (NE::LocString (L"Division"), position));
	});
	AddNodeTreeItem (nodeTree, arithmeticNodes, L"Floor", NUIE::IconId (11), [&] (const NUIE::Point& position) {
		return NUIE::UINodePtr (new BI::FloorNode (NE::LocString (L"Floor"), position));
	});
	AddNodeTreeItem (nodeTree, arithmeticNodes, L"Ceil", NUIE::IconId (12), [&] (const NUIE::Point& position) {
		return NUIE::UINodePtr (new BI::CeilNode (NE::LocString (L"Ceil"), position));
	});
	AddNodeTreeItem (nodeTree, arithmeticNodes, L"Abs", NUIE::IconId (13), [&] (const NUIE::Point& position) {
		return NUIE::UINodePtr (new BI::AbsNode (NE::LocString (L"Abs"), position));
	});
	AddNodeTreeItem (nodeTree, arithmeticNodes, L"Negative", NUIE::IconId (14), [&] (const NUIE::Point& position) {
		return NUIE::UINodePtr (new BI::NegativeNode (NE::LocString (L"Negative"), position));
	});
	AddNodeTreeItem (nodeTree, arithmeticNodes, L"Sqrt", NUIE::IconId (15), [&] (const NUIE::Point& position) {
		return NUIE::UINodePtr (new BI::SqrtNode (NE::LocString (L"Sqrt"), position));
	});
	size_t otherNodes = nodeTree.AddGroup (L"Other Nodes");
	AddNodeTreeItem (nodeTree, otherNodes, L"List Builder", NUIE::IconId (16), [&] (const NUIE::Point& position) {
		return NUIE::UINodePtr (new BI::ListBuilderNode (NE::LocString (L"List Builder"), position));
	});
	AddNodeTreeItem (nodeTree, otherNodes, L"Viewer", NUIE::IconId (17), [&] (const NUIE::Point& position) {
		return NUIE::UINodePtr (new BI::MultiLineViewerNode (NE::LocString (L"Viewer"), position, 5));
	});
}

AppEventHandler::AppEventHandler () :
	nodeTree (),
	nodeEditor (nullptr),
	nsView (nullptr)
{
	InitNodeTree (nodeTree);
}

AppEventHandler::~AppEventHandler ()
{
	
}

void AppEventHandler::Init (NUIE::NodeEditor* nodeEditorPtr, void* nsViewPtr)
{
	nodeEditor = nodeEditorPtr;
	nsView = nsViewPtr;
}

NUIE::MenuCommandPtr AppEventHandler::OnContextMenu (NUIE::EventHandler::ContextMenuType type, const NUIE::Point& position, const NUIE::MenuCommandStructure& commands)
{
	if (type == NUIE::EventHandler::ContextMenuType::EmptyArea) {
		NUIE::MenuCommandStructure finalCommands = commands;
		NUIE::AddNodeTreeToMenuStructure (nodeTree, position, nodeEditor, finalCommands);
		return MAS::SelectCommandFromContextMenu ((NSView*) nsView, position, finalCommands);
	} else {
		return MAS::SelectCommandFromContextMenu ((NSView*) nsView, position, commands);
	}
}

bool AppEventHandler::OnParameterSettings (NUIE::EventHandler::ParameterSettingsType type, NUIE::ParameterInterfacePtr paramAccessor)
{
	std::wstring title;
	if (type == NUIE::EventHandler::ParameterSettingsType::Node) {
		title = NE::LocalizeString (L"Node Parameters");
	} else if (type == NUIE::EventHandler::ParameterSettingsType::Group) {
		title = NE::LocalizeString (L"Group Parameters");
	} else {
		DBGBREAK ();
	}
	MAS::ParameterDialog paramDialog (title, paramAccessor);
	return paramDialog.Show (0, 0);
}

void AppEventHandler::OnDoubleClick (const NUIE::Point& position, NUIE::MouseButton mouseButton)
{
	#pragma unused (mouseButton)
	#pragma unused (position)
}

AppNodeUIEnvironment::AppNodeUIEnvironment () :
	stringConverter (NE::BasicStringConverter (MAS::GetStringSettingsFromSystem ())),
	skinParams (GetAppSkinParams ()),
	evaluationEnv (NE::EmptyEvaluationEnv),
	eventHandler (),
	clipboardHandler (),
	nodeEditorControl (CreateDrawingContext (ContextType::NSViewOffscreen))
{
	
}

AppNodeUIEnvironment::~AppNodeUIEnvironment ()
{

}

void AppNodeUIEnvironment::Init (NUIE::NodeEditor* nodeEditorPtr, void* nativeParentHandle, int x, int y, int width, int height)
{
	nodeEditorControl.Init (nodeEditorPtr, nativeParentHandle, x, y, width, height);
	eventHandler.Init (nodeEditorPtr, nativeParentHandle);
	// TODO: temporary
	nodeEditorPtr->AddNode (std::make_shared<BI::IntegerIncrementedNode> (NE::LocString (L"Integer Increment"), NUIE::Point (100, 100)));
	nodeEditorPtr->AddNode (std::make_shared<BI::AdditionNode> (NE::LocString (L"Addition"), NUIE::Point (300, 250)));
}

void AppNodeUIEnvironment::Resize (int x, int y, int width, int height)
{
	nodeEditorControl.Resize (x, y, width, height);
}

const NE::StringConverter& AppNodeUIEnvironment::GetStringConverter ()
{
	return stringConverter;
}

const NUIE::SkinParams& AppNodeUIEnvironment::GetSkinParams ()
{
	return skinParams;
}

NUIE::DrawingContext& AppNodeUIEnvironment::GetDrawingContext ()
{
	return nodeEditorControl.GetDrawingContext ();
}

double AppNodeUIEnvironment::GetWindowScale ()
{
	return 1.0;
}

NE::EvaluationEnv& AppNodeUIEnvironment::GetEvaluationEnv ()
{
	return evaluationEnv;
}

void AppNodeUIEnvironment::OnEvaluationBegin ()
{

}

void AppNodeUIEnvironment::OnEvaluationEnd ()
{

}

void AppNodeUIEnvironment::OnValuesRecalculated ()
{

}

void AppNodeUIEnvironment::OnRedrawRequested ()
{
	nodeEditorControl.Invalidate ();
}

NUIE::EventHandler& AppNodeUIEnvironment::GetEventHandler ()
{
	return eventHandler;
}

NUIE::ClipboardHandler& AppNodeUIEnvironment::GetClipboardHandler ()
{
	return clipboardHandler;
}

void AppNodeUIEnvironment::OnSelectionChanged (const NUIE::Selection&)
{

}

void AppNodeUIEnvironment::OnUndoStateChanged (const NUIE::UndoState&)
{
	
}

void AppNodeUIEnvironment::OnClipboardStateChanged (const NUIE::ClipboardState&)
{

}

void AppNodeUIEnvironment::OnIncompatibleVersionPasted (const NUIE::Version&)
{

}

Application::Application () :
	uiEnvironment (),
	nodeEditor (uiEnvironment)
{

}

void Application::Run ()
{
	NSAutoreleasePool* pool = [[NSAutoreleasePool alloc] init];
	NSApp = [NSApplication sharedApplication];

	NSRect windowRect = NSMakeRect (0.0f, 0.0f, 900.0f, 500.0f);
	NSWindow* myWindow = [
	[NSWindow alloc]
		initWithContentRect : windowRect
		styleMask:
			NSWindowStyleMaskTitled |
			NSWindowStyleMaskClosable |
			NSWindowStyleMaskMiniaturizable |
			NSWindowStyleMaskResizable
		backing : NSBackingStoreBuffered
		defer : NO
	];

	[myWindow setTitle : @"Visual Script Engine Demo"];

	WindowController* myWindowController = [[WindowController alloc] autorelease];
	NSView* contentView = [[[NSView alloc] initWithFrame : windowRect] autorelease];

	[myWindowController setWindow : myWindow];
	[myWindowController setUIEnvironment : &uiEnvironment];

	[myWindow setContentView : contentView];
	[myWindow setDelegate : myWindowController];

	NUIE::EnableFeature (NUIE::Feature::ConnectionDisplay, true);
	uiEnvironment.Init (&nodeEditor, contentView, 0, 0, windowRect.size.width, windowRect.size.height);
	[myWindow center];
	[myWindow makeKeyAndOrderFront : nil];

	[NSApp run];

	[NSApp release];
	[pool release];
}
